# PAM

# The Linux PAM package contains Pluggable Authentication Modules used by the local system administrator to control how application programs authenticate users.

# Contents
# Installed Programs: faillock, mkhomedir_helper, pam_namespace_helper, pam_timestamp_check, pwhistory_helper, unix_chkpwd and unix_update
# Installed Libraries: libpam.so, libpamc.so and libpam_misc.so
# Installed Directories: /etc/security, /usr/lib/security, /usr/include/security and /usr/share/doc/Linux-PAM-1.6.1

# Short Descriptions
# faillock - displays and modifies the authentication failure record files
# mkhomedir_helper - is a helper binary that creates home directories
# pam_namespace_helper - is a helper program used to configure a private namespace for a user session
# pwhistory_helper - is a helper program that transfers password hashes from passwd or shadow to opasswd
# pam_timestamp_check - is used to check if the default timestamp is valid
# unix_chkpwd - is a helper binary that verifies the password of the current user
# unix_update - is a helper binary that updates the password of a given user
# libpam.so - provides the interfaces between applications and the PAM modules

# Kernel Configuration

LINUX_VERSION=$(uname -r)

if [[ ! $(cat /boot/config-$LINUX_VERSION | grep "CONFIG_AUDIT=y") ]]; then 
	echo "ERROR: Kernel needs built with 'General setup ---> [*] Auditing support [AUDIT]'"
	exit -1
fi

# The shipped libtool.m4 file has a configuration inconsistent with LFS /usr hierarchy. This issue would cause libpam_misc.so linked with an rpath flag which may sometimes cause troubles or even security issues. Regenerate the building system to fix the inconsistency:
autoreconf -fi

# If you downloaded the documentation, unpack the tarball by issuing the following command.
tar -xf ../$(basename $DOC_LINUX_PAM) --strip-components=1


if [[ -f ../$(basename $PATCH_LINUX_PAM) ]]; then
	patch -Np1 -i ../$(basename $PATCH_LINUX_PAM)
fi

LINUX_PAM_VERSION=$((basename $PKG_LINUX_PAM .tar.xz) | cut -d "-" -f 2)

# Compile and link Linux PAM by running the following commands:
./configure --prefix=/usr                        \
            --sbindir=/usr/sbin                  \
            --sysconfdir=/etc                    \
            --libdir=/usr/lib                    \
            --enable-securedir=/usr/lib/security \
            --docdir=/usr/share/doc/Linux-PAM-$LINUX_PAM_VERSION
make
  
install -v -m755 -d /etc/pam.d

if $RUN_TESTS
then
# To test the results, a suitable /etc/pam.d/other configuration file must exist.
cat << EOF > /etc/pam.d/other
auth     required       pam_deny.so
account  required       pam_deny.so
password required       pam_deny.so
session  required       pam_deny.so
EOF

    set +e
    make check
    set -e
	# For a first-time installation, remove the configuration file created earlier by issuing the following command as the root user:
	rm -fv /etc/pam.d/other
fi

# You should also be aware that make install overwrites the configuration files in /etc/security as well as /etc/environment. If you have modified those files, be sure to back them up.
if [ -f /etc/security ]; then
	cp /etc/security /etc/security.archive-$(date +'%Y%m%d%H%M')
fi

if [ -f /etc/environment ]; then
	cp /etc/environment /etc/environment.archive-$(date +'%Y%m%d%H%M')
fi

# As Root:

make install
chmod -v 4755 /usr/sbin/unix_chkpwd

install -vdm755 /etc/pam.d &&
cat > /etc/pam.d/system-account << "EOF" &&
# Begin /etc/pam.d/system-account

account   required    pam_unix.so

# End /etc/pam.d/system-account
EOF

cat > /etc/pam.d/system-auth << "EOF" &&
# Begin /etc/pam.d/system-auth

auth      required    pam_unix.so

# End /etc/pam.d/system-auth
EOF

cat > /etc/pam.d/system-session << "EOF" &&
# Begin /etc/pam.d/system-session

session   required    pam_unix.so

# End /etc/pam.d/system-session
EOF

cat > /etc/pam.d/system-password << "EOF"
# Begin /etc/pam.d/system-password

# use yescrypt hash for encryption, use shadow, and try to use any
# previously defined authentication token (chosen password) set by any
# prior module.
password  required    pam_unix.so       yescrypt shadow try_first_pass

# End /etc/pam.d/system-password
EOF

cat > /etc/pam.d/other << "EOF"
# Begin /etc/pam.d/other

auth        required        pam_warn.so
auth        required        pam_deny.so
account     required        pam_warn.so
account     required        pam_deny.so
password    required        pam_warn.so
password    required        pam_deny.so
session     required        pam_warn.so
session     required        pam_deny.so

# End /etc/pam.d/other
EOF

# You should now reinstall the Shadow-4.16.0 and Systemd-256.4 packages.

update_version_list $PKG_LINUX_PAM
